import 'package:rad/rad.dart';
import 'package:uuid/uuid.dart';

import 'package:photogram_admin_cp/import/core.dart';
import 'package:photogram_admin_cp/import/bloc.dart';
import 'package:photogram_admin_cp/import/data.dart';

class AppActiveContent {
  final BuildContext context;
  late final AppContentProvider _appContentProvider;
  late final String _widgetInstanceId;

  AppActiveContent(this.context) {
    _appContentProvider = AppProvider.of(context).appContentProvider;
    _widgetInstanceId = const Uuid().v1();
  }

  void clear() => _appContentProvider.clearWidget(disposedWidgetInstanceId: _widgetInstanceId);

  void dispose() => _appContentProvider.disposeWidget(disposedWidgetInstanceId: _widgetInstanceId);

  Map<int, T> pagedModels<T>() => _appContentProvider.pagedModels<T>(widgetInstanceId: _widgetInstanceId);

  T? watch<T>(int contentId) => _appContentProvider.watch<T>(
        contentId: contentId,
        widgetInstanceId: _widgetInstanceId,
      );

  T? read<T>(int contentId) => _appContentProvider.read<T>(
        contentId: contentId,
        widgetInstanceId: _widgetInstanceId,
      );

  void unlink<T>(int contentId) => _appContentProvider.unlink<T>(
        contentId: contentId,
        widgetInstanceId: _widgetInstanceId,
      );

  List<int> addOrUpdateListOfModels<T>({List<T> models = const []}) => _appContentProvider.addOrUpdateAll<T>(
        models: models,
        widgetInstanceId: _widgetInstanceId,
      );

  int addOrUpdateModel<T>(T model) => _appContentProvider.addOrUpdateModel<T>(
        model: model,
        widgetInstanceId: _widgetInstanceId,
      );

  /*
  |--------------------------------------------------------------------------
  | blocs
  |--------------------------------------------------------------------------
  */

  AuthBloc get authBloc => AppProvider.of(context).auth;

  ApiRepository get apiRepository => AppProvider.of(context).apiRepo;

  /*
  |--------------------------------------------------------------------------
  | handle content
  | 
  | This is our global port that handle most of the responses. Having all 
  | parsing at one place is easier to manage.
  | 
  | !! Important Note !!
  | 
  | This piece of code looks like a lengthy call but it's not. This get's 
  | called to collect data from response. Response part of all requests can
  | contains maximum of 4 different types of models, which means on each call
  | only 4 blocks from all below are going to execute and rest will be skipped
  | Skip is effecient and involves a simple hash map look up. We can easily 
  | introduce concurrency in this part, if it ever needs to be. 
  |--------------------------------------------------------------------------
  */

  void handleResponse(ResponseModel responseModel) => _captureData(responseModel);

  void _captureData(ResponseModel responseModel) {
    try {
      if (END_OF_RESULTS_MSG != responseModel.message) {
        /*
        |--------------------------------------------------------------------------
        | check for users
        |--------------------------------------------------------------------------
        */

        var userModelsToAdd = <int, UserModel>{};

        if (responseModel.content.containsKey(UserTable.tableName)) {
          for (var dataMap in (responseModel.content[UserTable.tableName] as List)) {
            var model = UserModel.fromJson(dataMap);

            if (model.isModel) {
              userModelsToAdd.addAll({model.intId: model});
            }
          }
        }

        /*
        |--------------------------------------------------------------------------
        | check for admins
        |--------------------------------------------------------------------------
        */

        var adminModelsToAdd = <int, AdminModel>{};

        if (responseModel.content.containsKey(AdminTable.tableName)) {
          for (var dataMap in (responseModel.content[AdminTable.tableName] as List)) {
            var model = AdminModel.fromJson(dataMap);

            if (model.isModel) {
              adminModelsToAdd.addAll({model.intId: model});
            }
          }
        }

        /*
        |--------------------------------------------------------------------------
        | check for settings
        |--------------------------------------------------------------------------
        */

        var settingModelsToAdd = <int, SettingModel>{};

        if (responseModel.content.containsKey(SettingTable.tableName)) {
          for (var dataMap in (responseModel.content[SettingTable.tableName] as List)) {
            var model = SettingModel.fromJson(dataMap);

            if (model.isModel) {
              settingModelsToAdd.addAll({model.intId: model});
            }
          }
        }

        /*
        |--------------------------------------------------------------------------
        | push new active content
        |--------------------------------------------------------------------------
        */

        ///
        /// ! helpful warning: if you're modifying source code
        ///
        /// whenever you add a new model type here, make sure  to add entires in garbage collector
        /// which exists entirely in its logical form, in the module [AppContentProvider]. not doing
        /// that directly lead to memory leaks. end user might not notice that but if a user happens
        /// to stay in app for long time and keeps on loading new content, then, over time, this gonna
        /// adds up and can eventually lead app to crash to prevent such memory leaks make sure that
        /// for each model type, there exists:
        ///
        /// 1. atleast one entry in [AppContentProvider._clearWidgetDependencies]
        /// 2. atleast one entry in [AppContentProvider._releaseResources]
        ///
        /// I wish I could automate this process for you but Dart doesn't support dynamic binding
        /// for type parameters, which means there's no way I can keep all types at one place and
        /// run a generic call for all model types.
        ///
        /// - erlage, Feb 8
        ///

        if (userModelsToAdd.isNotEmpty) {
          addOrUpdateListOfModels<UserModel>(models: userModelsToAdd.values.toList());
        }

        if (adminModelsToAdd.isNotEmpty) {
          addOrUpdateListOfModels<AdminModel>(models: adminModelsToAdd.values.toList());
        }

        if (settingModelsToAdd.isNotEmpty) {
          addOrUpdateListOfModels<SettingModel>(models: settingModelsToAdd.values.toList());
        }
      }
    } catch (e) {
      AppLogger.exception(e);
    }
  }
}
